#include "stdafx.h"
#include <float.h>
#include <math.h>
#include <string.h>
#include "Equation.h"

/* These defines only happen if the values are not already defined!  You may
   want to add more precision here, if your machine supports it. */
#ifndef M_PI
#define M_PI    3.14159265358979323846
#endif
#ifndef M_E
#define M_E     2.71828182845904523536
#endif
#ifndef M_2PI
#define M_2PI     6.283185307
#endif
#ifndef M_RADIAN
#define M_RADIAN  57.29577951
#endif
#ifndef M_K_BOLTZMANN
#define M_K_BOLTZMANN   1.3806488e-23
#endif
#ifndef M_Q_ELECTRON
#define M_Q_ELECTRON  1.602176565e-19
#endif
#ifndef M_C_LIGHT
#define M_C_LIGHT  2.99792458e8
#endif
#ifndef M_H_PLANK
#define M_H_PLANK  6.62606957e-34
#endif
#ifndef M_U0_MAGNETIC
#define M_U0_MAGNETIC  12.566370614e-7
#endif
#ifndef M_E0_ELECTRIC
#define M_E0_ELECTRIC  8.85418781e-12
#endif



#pragma warning(disable:4996) 

bool ExtractSpiceModelWords(char **line_in, int num_strings);
void DeleteCharArray(char **array, int n);
char **NewCharArray(int n, int size);
int CountCharacters(CString CText, char character);
double MKSStringToFloat(CString &CText);

double Rand(double seed);

/*************************************************************************
**                                                                       **
** PROTOTYPES FOR CUSTOM MATH FUNCTIONS                                  **
**                                                                       **
 *************************************************************************/

double deg( double x );
double rad( double x );
double db(double x);
double deg( double x );
double rad( double x );
double asinh(double x);
double acosh(double x);
double atanh(double x);
/*************************************************************************
**                                                                       **
** VARIABLE DECLARATIONS                                                 **
**                                                                       **
 *************************************************************************/

//int   ERROR;                 /* The error number */
//char  ERTOK[TOKLEN + 1];     /* The token that generated the error */
//int   ERPOS;                 /* The offset from the start of the expression */
char* ERANC;                 /* Used to calculate ERPOS */

/*
   Add any "constants" here...  These are "read-only" values that are
   provided as a convienence to the user.  Their values can not be
   permanently changed.  The first field is the variable name, the second
   is its value.
*/
VARIABLE Consts[] =
{
   /* name, value */
   { "pi",      M_PI },
   { "2pi",     M_2PI },
   { "radian",     M_RADIAN },
   { "k_boltzmann", M_K_BOLTZMANN},
   { "q_electron", M_Q_ELECTRON },
   { "c_light", M_C_LIGHT },
   { "h_plank", M_H_PLANK },
   { "mu_0", M_U0_MAGNETIC },
   { "e_0", M_E0_ELECTRIC },

   {"\0"}, // k.a 21 Dec 2014, fatal crash prevention as this value is tested for its NULL value
//  { 0 }
};

/*
   Add any math functions that you wish to recognise here...  The first
   field is the name of the function as it would appear in an expression.
   The second field tells how many arguments to expect.  The third is
   a pointer to the actual function to use.
*/
FUNCTION Funcs[] =
{
   /* name, function to call */
   { "sin",     1,    sin },
   { "cos",     1,    cos },
   { "tan",     1,    tan },
   { "asin",    1,    asin },
   { "acos",    1,    acos },
   { "atan",    1,    atan },
   { "sinh",    1,    sinh },
   { "cosh",    1,    cosh },
   { "tanh",    1,    tanh },
   { "asinh",    1,    asinh },
   { "acosh",    1,    acosh },
   { "atanh",    1,    atanh },
   { "exp",     1,    exp },
   { "log",     1,    log10 },
   { "ln",      1,    log },
   { "sqrt",    1,    sqrt },
   { "floor",   1,    floor },
   { "ceil",    1,    ceil },
   { "abs",     1,    fabs },
   { "deg",     1,    deg },
   { "rad",     1,    rad },
   { "rand",     1,    Rand },
   {"db", 1, db},
   {"\0"}, // k.a 21 Dec 2014, fatal crash prevention as this value is tested for its NULL value
// { 0 }
};

VARIABLE        Vars[MAXVARS];       /* Array for user-defined variables */
unsigned char*  expression;          /* Pointer to the user's expression */
unsigned char   token[TOKLEN + 1];   /* Holds the current token */
int             type;                /* Type of the current token */


/*************************************************************************
**                                                                       **
** Some custom math functions...   Note that they must be prototyped     **
** above (if your compiler requires it)                                  **
**                                                                       **
** deg( x )             Converts x radians to degrees.                   **
** rad( x )             Converts x degrees to radians.                   **
**                                                                       **
 *************************************************************************/

double deg( double x )
{
   return( x * 180.0 / M_PI );
}

double rad( double x )
{
   return( x * M_PI / 180.0 );
}

double db( double x )
{
	if(!x) return 0;

   return( 20*log10(x) );
}

double asinh( double x )
{
	double y;

	y = x + sqrt(x*x+1);

   return( y );
}

double acosh( double x )
{
	double y;

	if(x < 1) return 0;

	y = x + sqrt(x*x-1);

   return( log(y) );
}

double atanh( double x )
{
	double y;

	if(!x) return 0;

	if(abs(x) < 1 ) return 0;

	y = sqrt((1+x)/(1-x));

   return( log(y)/2 );
}

double Rand(double seed)
{
	if(!seed)  return (double)rand()/(double)RAND_MAX;
		
	srand((unsigned)time(NULL));

	return (double)rand()/(double)RAND_MAX;
}
/*************************************************************************
**                                                                       **
** ClearAllVars()                                                        **
**                                                                       **
** Erases all user-defined variables from memory. Note that constants    **
** can not be erased or modified in any way by the user.                 **
**                                                                       **
** Returns nothing.                                                      **
**                                                                       **
 *************************************************************************/

void ClearAllVars(void)
{
   int i;

   for(i = 0; i < MAXVARS; i++ )
   {
	   VARIABLE &var = Vars[i];

      *var.name = 0;
      var.value = 0;
   }
}


/*************************************************************************
**                                                                       **
** ClearVar( char* name )                                                **
**                                                                       **
** Erases the user-defined variable that is called NAME from memory.     **
** Note that constants are not affected.                                 **
**                                                                       **
** Returns 1 if the variable was found and erased, or 0 if it didn't     **
** exist.                                                                **
**                                                                       **
 *************************************************************************/

int ClearVar( char* name )
{
   int i;

   for( i = 0; i < MAXVARS; i++ )
   {
		VARIABLE &var = Vars[i];

		if( *var.name && ! strcmp( name, var.name ) )
		{
			*var.name = 0;
			var.value = 0;

			return 1;
		}
   }
   return 0;
}


/*************************************************************************
**                                                                       **
** GetValue( char* name, TYPE* value )                                   **
**                                                                       **
** Looks up the specified variable (or constant) known as NAME and       **
** returns its contents in VALUE.                                        **
**                                                                       **
** First the user-defined variables are searched, then the constants are **
** searched.                                                             **
**                                                                       **
** Returns 1 if the value was found, or 0 if it wasn't.                  **
**                                                                       **
 *************************************************************************/

int GetValue( char* name, TYPE* value )
{
   int i;

   /* Now check the user-defined variables. */
   for( i = 0; i < MAXVARS; i++ )
   {
	  VARIABLE &var = Vars[i];

      if( *var.name && ! strcmp( name, var.name ) )
      {
         *value = var.value;

         return 1;
      }
   }

   /* Now check the programmer-defined constants. */
   for( i = 0; *Consts[i].name; i++ )
   {
	   VARIABLE &Cons = Consts[i];

      if( *Cons.name && ! strcmp( name, Cons.name ) )
      {
         *value = Cons.value;

         return 1 ;
      }
   }

   return 0 ;
}


/*************************************************************************
**                                                                       **
** SetValue( char* name, TYPE* value )                                   **
**                                                                       **
** First, it erases any user-defined variable that is called NAME.  Then **
** it creates a new variable called NAME and gives it the value VALUE.   **
**                                                                       **
** Returns 1 if the value was added, or 0 if there was no more room.     **
**                                                                       **
 *************************************************************************/

int SetValue( char* name, TYPE* value )
{
 //  char b[30];
   int  i;

   ClearVar( name );

   for( i = 0; i < MAXVARS; i++ )
   {
		VARIABLE &var = Vars[i];

		if( ! *var.name )
		{
			strcpy( var.name, name );
			var.name[VARLEN] = 0;
			var.value = *value;

			return( 1 );
		}
   }

   return( 0 );
}


/*************************************************************************
**                                                                       **
** Parse()   Internal use only                                           **
**                                                                       **
** This function is used to grab the next token from the expression that **
** is being evaluated.                                                   **
**                                                                       **
 *************************************************************************/

void Parse(void)
{
   char* t;

   type = 0;
   t = (char*)token;
   int mks_flag;
   int sci_notation_flag;
   int plus_minus_flag;

   while( iswhite( *expression ) )
      expression++;
   if( isdelim( *expression ) )
   {
      type = DEL;
      *t++ = *expression++;
   }
   else if( isnumer( *expression ) )
   {
      type = NUM;

	  while(isnumer(*expression)) 
		  *t++ = *expression++;

	  // added kevin aylward for MKS p, u, k, M support Nov 2014
	  mks_flag = (*expression == 'f') || (*expression == 'F') || (*expression == 'p') || (*expression == 'P')
				|| (*expression == 'n') || (*expression == 'N') || (*expression == 'u') || (*expression == 'U') || (*expression == 'm')
				|| (*expression == 'R') || (*expression == 'r') || (*expression == 'V') || (*expression == 'v') || (*expression == 'K')
				|| (*expression == 'k') || (*expression == 'M') || (*expression == 'G') || (*expression == 'g') || (*expression == 'T')
				|| (*expression == 't');

	  sci_notation_flag = (*expression == 'e'); 

//	  if(*expression == 'e') // added kevin aylward for sci notation support
	  if(mks_flag || sci_notation_flag) // added kevin aylward for MKS p, u, k, M support Nov 2014
	  {
		  *t++ = *expression++;// 

		  if(sci_notation_flag)// Grab all after the "e", 1.23e+010 to pass to string to float conversion
		  {
			  plus_minus_flag = 1; // only include 1st - or +, any after is an operator

				while(isnumer(*expression) || (*expression == '+' || *expression == '-') && plus_minus_flag)
				{
					if(*expression == '+' || *expression == '-') plus_minus_flag = 0;

						*t++ = *expression++; // include next + or -, 1.23e+002
				}
		  }
		  else // Assume any number after MKS multiplier is not needed. Required to fix 3.2m+4.5m failing for the +
		  {
				while(isnumer(*expression)) 
					*t++ = *expression++;
		  }  
	  }
   }
   else if( isalpha( *expression ) )
   {
      type = VAR;
      while( isalpha( *expression ) )
        *t++ = *expression++;
      token[VARLEN] = 0;
   }
   else if( *expression )
   {
      *t++ = *expression++;
      *t = 0;
	  return;//    ERR( E_SYNTAX );
   }
   *t = 0;
   while( iswhite( *expression ) )
      expression++;
}


/*************************************************************************
**                                                                       **
** Level1( TYPE* r )   Internal use only                                 **
**                                                                       **
** This function handles any variable assignment operations.             **
** It returns a value of 1 if it is a top-level assignment operation,    **
** otherwise it returns 0                                                **
**                                                                       **
 *************************************************************************/

int Level1( TYPE* r )
{
   char t[VARLEN + 1];

   if( type == VAR )
      if( *expression == '=' )
      {
         strcpy( t, (char*)token );
         Parse();
         Parse();
         if( !*token )
         {
            ClearVar( t );
            return(1);
         }
         Level2( r );
         if( ! SetValue( t, r ) )
           return 0;// ERR( E_MAXVARS );
         return( 1 );
      }
   Level2( r );
   return( 0 );
}


/*************************************************************************
**                                                                       **
** Level2( TYPE* r )   Internal use only                                 **
**                                                                       **
** This function handles any addition and subtraction operations.        **
**                                                                       **
 *************************************************************************/

void Level2( TYPE* r )
{
   TYPE t = 0;
   char o;

   Level3( r );
   while( (o = *token) == '+' || o == '-' )
   {
      Parse();
      Level3( &t );
      if( o == '+' )
         *r = *r + t;
      else if( o == '-' )
         *r = *r - t;
   }
}


/*************************************************************************
**                                                                       **
** Level3( TYPE* r )   Internal use only                                 **
**                                                                       **
** This function handles any multiplication, division, or modulo.        **
**                                                                       **
 *************************************************************************/

void Level3( TYPE* r )
{
   TYPE t;
   char o;

   Level4( r );
   while( (o = *token) == '*' || o == '/' || o == '%' )
   {
      Parse();
      Level4( &t );
      if( o == '*' )
         *r = *r * t;
      else if( o == '/' )
      {
         if( t == 0 )
           t=1;// ERR( E_DIVZERO );
         *r = *r / t;
      }
      else if( o == '%' )
      {
         if( t == 0 )
            t=1;//ERR( E_DIVZERO );
         *r = fmod( *r, t );
      }
   }
}


/*************************************************************************
**                                                                       **
** Level4( TYPE* r )   Internal use only                                 **
**                                                                       **
** This function handles any "to the power of" operations.               **
**                                                                       **
 *************************************************************************/

void Level4( TYPE* r )
{
   TYPE t;

   Level5( r );
   if( *token == '^' )
   {
      Parse();
      Level5( &t );
      *r = pow( *r, t );
   }
}


/*************************************************************************
**                                                                       **
** Level5( TYPE* r )   Internal use only                                 **
**                                                                       **
** This function handles any unary + or - signs.                         **
**                                                                       **
 *************************************************************************/

void Level5( TYPE* r )
{
   char o = 0;

   if( *token == '+' || *token == '-' )
   {
      o = *token;
      Parse();
   }
   Level6( r );
   if( o == '-' )
      *r = -*r;
}


/*************************************************************************
**                                                                       **
** Level6( TYPE* r )   Internal use only                                 **
**                                                                       **
** This function handles any literal numbers, variables, or functions.   **
**                                                                       **
 *************************************************************************/

void Level6( TYPE* r )
{
   int  i;
   int  n;
   TYPE a[3];
   CString CNumberText;

   if( *token == '(' )
   {
      Parse();
      if( *token == ')' )
         return;//ERR( E_NOARG );
      Level1( r );
      if( *token != ')' )
         return;//ERR( E_UNBALAN );
      Parse();
   }
   else
   {
      if( type == NUM )
      {
 //        *r = (TYPE) atof( (char*)token ); // support for spice MKS p, u, k, M multipliers kevin Nov 2014

		  CNumberText = token;

		  *r = MKSStringToFloat(CNumberText);

         Parse();
      }
      else if( type == VAR )
      {
         if( *expression == '(' )
         {
            for( i = 0; *Funcs[i].name; i++ )
               if( ! strcmp( (char*)token, Funcs[i].name ) )
               {
                  Parse();
                  n = 0;
                  do
                  {
                     Parse();
                     if( *token == ')' || *token == ',' )
                        return;//ERR( E_NOARG );
                     a[n] = 0;
                     Level1( &a[n] );
                     n++;
                  } while( n < 4 && *token == ',' );
                  Parse();
                  if( n != Funcs[i].args )
                  {
                     strcpy((char*) token, Funcs[i].name );
                     return;//ERR( E_NUMARGS );
                  }
                  //*r = Funcs[i].func( a[0], a[1], a[2] );

				  *r = Funcs[i].func( a[0]);// KA changed

                  return;
               }
               if( ! *Funcs[i].name )
                 return;// ERR( E_BADFUNC );
            }
            else if( ! GetValue((char*) token, r ) )
               return;//ERR( E_UNKNOWN );
         Parse();
      }
      else
         return;//ERR( E_SYNTAX );
   }
}


/*************************************************************************
**                                                                       **
** Evaluate( char* e, TYPE* result, int* a )                             **
**                                                                       **
** This function is called to evaluate the expression E and return the   **
** answer in RESULT.  If the expression was a top-level assignment, a    **
** value of 1 will be returned in A, otherwise it will contain 0.        **
**                                                                       **
** Returns E_OK if the expression is valid, or an error code.            **
**                                                                       **
 *************************************************************************/

int Evaluate( char* e, TYPE* result, int* a )
{
   expression = (unsigned char*)e;
   ERANC = e;

//   strlwr((char*) expression ); // needs to be case sensitive for M=1e6 Dec 9th 2014

   *result = 0;

   Parse();

   if( ! *token )
      return 0;//ERR( E_EMPTY );
   *a = Level1( result );
   return( E_OK );
}


int Equation(CString CExpression, CString CValues, double *data_result)
{
	int num_param	= CountCharacters(CValues, '=');
	int size_words	= CValues.GetLength() + 1;
	char** words;

	int num_words = num_param * 2;

	data_result = 0;

	words = NewCharArray(1 + num_words, size_words);

	if(!words) return false;

	strcpy(words[0], CValues.GetBuffer(size_words));

	ExtractSpiceModelWords(words, num_words);

	int count2 = num_words/2;

	char **p_data = &words[0];

	CString CName, CValue;

	double dummy;
	int f_flag;

	for(int p = 0; p < count2; p++)
	{
		CName = *++p_data;

		if(CName == "") continue;

		CValue = *++p_data;

		if(CValue == "") continue;

		CName = CName + '=' + CValue;//x=1.2

		try
		{
			if(Evaluate(CName.GetBuffer(CName.GetLength() + 1), &dummy, &f_flag) != E_OK) 
			{
				DeleteCharArray(words, 1 + num_words);

				return false;
			} 
		}
		catch(...)
		{
			_fpreset();

			DeleteCharArray(words, 1 + num_words);

			continue;
		}
	}

	DeleteCharArray(words, 1 + num_words);	

	try
	{
		if(Evaluate(CExpression.GetBuffer(CExpression.GetLength() + 1), data_result, &f_flag) == E_OK) 
		{
			return true;
		}
	}
	catch (...)
	{
		_fpreset();

		DeleteCharArray(words, 1 + num_words);

		return false;
	}

	return false;
} 

int Equation(CString CExpression, double *data_result)
{
	int f_flag;

	try // catch div by zero etc.
	{
		if(Evaluate(CExpression.GetBuffer(CExpression.GetLength() + 1), data_result, &f_flag) == E_OK) 
		{
			return true;
		}
	}
	catch(...)
	{
		_fpreset();

		return false;
	}

	return false;
}

int EquationSetVarible(CString CName, CString CValue)
{
	double dummy;
	int f_flag;

	CName = CName + '=' + CValue;

	try
	{
		if(Evaluate(CName.GetBuffer(CName.GetLength() + 1), &dummy, &f_flag) == E_OK) return true;
	}
	catch(...)
	{
		_fpreset();

		return false;
	}

	return false;
}

int EquationSetVaribles(CString CNameEqualsValueList)
{
	int num_param	= CountCharacters(CNameEqualsValueList, '=');
	int size_words	= CNameEqualsValueList.GetLength() + 1;
	char** words;
	double data;

	double dummy;
	int f_flag;
	
	int num_words = num_param * 2;

	words = NewCharArray(1 + num_words, size_words);

	if(!words) return false;

	strcpy(words[0], CNameEqualsValueList.GetBuffer(size_words));

	ExtractSpiceModelWords(words, num_words);

	int count2 = num_words/2;

	char **p_data = &words[0];

	CString CName, CValue;

	for(int p = 0; p < count2; p++)
	{
		CName = *++p_data;

		if(CName == "") continue;

		CValue = *++p_data;

		if(CValue == "") continue;

		data = MKSStringToFloat(CValue);
		
		CValue.Format("%e", data);

		CName = CName + '=' + CValue;

		try
		{
			if(Evaluate(CName.GetBuffer(CName.GetLength() + 1), &dummy, &f_flag) != E_OK) 
			{
				DeleteCharArray(words, 1 + num_words);

				return false;
			} 
		}
		catch(...)
		{
			_fpreset();

			DeleteCharArray(words, 1 + num_words);

			return false;
		}
	}

	DeleteCharArray(words, 1 + num_words);

	return true;
}

/*

  
*************************************************************************
**                                                                       **
** What follows is a main() routine that evaluates the command line      **
** arguments one at a time, and displays the results of each expression. **
** Without arguments, it becomes an interactive calculator.              **
**                                                                       **
 *************************************************************************


char* ErrMsgs[] =
{
   "Syntax error",
   "Unbalanced parenthesis",
   "Division by zero",
   "Unknown variable",
   "Maximum variables exceeded",
   "Unrecognised function",
   "Wrong number of arguments to function",
   "Missing an argument",
   "Empty expression"
};

main( int argc, char* argv[] )
{
   TYPE  result;
   int   i;
   int   ec;
   int   a;
   char  line[1024];
   char* v;

   ClearAllVars();
   // If we have command line arguments then we evaluate them and exit. 
   if( argc > 1 )
   {
      // Concatenate all arguments into one string. 
      strcpy( line, argv[1] );
      for( i = 2; i < argc; i++ )
         strcat( line, argv[i] );

      // Call the evaluator. 
      if( (ec = Evaluate( line, &result, &a )) == E_OK )
      {
         // If we didn't assign a variable, then print the result. 
         if( ! a )
            printf( "%g\n", result );
      }
      else if( ec != E_EMPTY )
      {
         // Display error info.  In this example, an E_EMPTY error is ignored. 
         printf( "ERROR: %s - %s", ErrMsgs[ERROR - 1], ERTOK );
         printf( "\n%s", ERANC );
         printf( "\n%*s^\n", ERPOS, "" );
      }
      return;
   }

   // There were no command line arguments, so lets go interactive. 
   printf( "\nEE - Equation Evaluator" );
   printf( "\nBy Mark Morley  December 1991" );
   printf( "\nEnter EXIT to quit.\n" );
   printf( "\nEE> " );

   // Input one line at a time from the user.  Note that it uses stdin, so on
      DOS or UNIX you could pipe a list of expressions into it... 
   for( gets( line ); !feof( stdin ); gets( line ) )
   {
      strlwr( line );

      // Did the user ask to exit? 
      if( ! strcmp( line, "exit" ) )
         return;

      // Did the user ask to see the variables in memory? 
      else if( ! strcmp( line, "vars" ) )
      {
         for( i = 0; i < MAXVARS; i++ )
            if( *Vars[i].name )
               printf( "%s = %g\n", Vars[i].name, Vars[i].value );
      }

      // Did the user ask to see the constants in memory? 
      else if( ! strcmp( line, "cons" ) )
      {
         for( i = 0; *Consts[i].name; i++ )
            printf( "%s = %g\n", Consts[i].name, Consts[i].value );
      }

      // Did the user ask to clear all variables? 
      else if( ! strcmp( line, "clr" ) )
         ClearAllVars();

      // If none of the above, then we attempt to evaluate the user's input. 
      else
      {
         // Call the evaluator. 
         if( (ec = Evaluate( line, &result, &a )) == E_OK )
         {
            // Only display the result if it was not an assignment. 
            if( ! a )
               printf( "%g\n", result );
         }
         else if( ec != E_EMPTY )
         {
            // Display error information.  E_EMPTY is ignored. 
            printf( "ERROR: %s - %s", ErrMsgs[ERROR - 1], ERTOK );
            printf( "\n%s", ERANC );
            printf( "\n%*s^\n", ERPOS, "" );
         }
      }
      printf( "EE> " );
   }
}
*/